const express = require("express");
const cors = require("cors");
const bodyParser = require("body-parser");
const morgan = require("morgan");
const http = require("http");
const path = require("path");
const helmet = require("helmet"); // <--- add helmet
const colors = require("colors");

const { logger, errorHandler } = require("./helper/logger.helper");
const api = require("./routes/router.js");
const { PORT } = require("./config");
const app = express();
const server = http.createServer(app);

// ***** SECURITY & LOGGING MIDDLEWARE *****

// Helmet (tune policies so it doesn't break your SPA/assets)
app.use(
  helmet({
    crossOriginResourcePolicy: false,
    crossOriginEmbedderPolicy: false,
    crossOriginOpenerPolicy: false,
  })
);

// List of allowed frontend origins
const allowedOrigins = [
  "http://localhost:5173",
  "http://localhost:5174",
  "https://leds-shanfoods.oee-tracker.com",
  "https://opex3-shanfoods.oee-tracker.com",
];

// CORS config with multiple origins and exposed header
app.use(
  cors({
    origin: function (origin, callback) {
      // Allow non-browser tools (like Postman) where origin is undefined
      if (!origin) return callback(null, true);
      if (allowedOrigins.includes(origin)) {
        return callback(null, true);
      }
      return callback(new Error("Not allowed by CORS"));
    },
    credentials: true,
    exposedHeaders: ["new-token"], // <--- THIS is key for your interceptor
  })
);

// Handle preflight for all routes (in case you need it)
app.use(morgan("dev"));
app.use(bodyParser.urlencoded({ extended: false, limit: "5mb" }));
app.use(bodyParser.json({ extended: false, limit: "5mb" }));
app.use(express.static("public"));

// ***** ROUTES *****
app.use("/api", api);

app.use(
  "/",
  express.static(path.resolve(path.join(__dirname, "./client/dist")))
);

app.use(
  "*",
  express.static(path.resolve(path.join(__dirname, "./client/dist")))
);

// ***** ERROR HANDLING *****
process.on("unhandledRejection", (error) => {
  console.error("Uncaught Promise Rejection:", error.stack);
  logger.error("Uncaught Promise Rejection:", error);
});

app.use(errorHandler);

server.listen(PORT, () => {
  console.log(
    `Server is running on: http://localhost:${PORT}`.bgCyan.green.bold
  );
});
