const { sendErrorResponse, sendResponse } = require("../utils");
const { performQuery } = require("../utils/dbFunctions");
const { safeNumber, formatDate, calculateOEE, groupBy } = require("../helper/calculation/calculation.helper");
const tables = require("../utils/tables");
const moment = require("moment-timezone");

const fetchOrderDetails = async (apiResponse) => {
  try {
    const fetchorderDetails = await performQuery(`
        SELECT 
            odd.id, odd.line_id, l.line, odd.order_id, o.order_no, odd.shift_id, sh.shift_name, 
            odd.order_start, odd.order_end, odd.production, odd.scrap, odd.shipable_product, 
            o.expected_qty, o.qty_unit, o.status AS order_status,
            p.product_code, p.product_name, p.speed AS product_speed
        FROM ${tables.order_details} odd
        LEFT JOIN ${tables.shift} sh ON odd.shift_id = sh.id
        LEFT JOIN ${tables.orders} o ON odd.order_id = o.id
        LEFT JOIN ${tables.line} l ON odd.line_id = l.id
        LEFT JOIN ${tables.product} p ON o.product_id = p.id
      where (
        odd.line_id = ? AND
            (
                STR_TO_DATE(odd.order_start,'%Y-%m-%d %H:%i:%s') >= STR_TO_DATE(?,'%Y-%m-%d %H:%i:%s') AND
                STR_TO_DATE(odd.order_start,'%Y-%m-%d %H:%i:%s') <= STR_TO_DATE(?,'%Y-%m-%d %H:%i:%s')
            )
      ) ORDER BY order_start ASC`,
      [apiResponse.line_id, apiResponse.start_date, apiResponse.end_date]
    );

    return fetchorderDetails;
  } catch (error) {
    console.log("error in fetching order details: ", error);
  }
};

const fetchEvents = async (apiResponse) => {
  try {
    const fetchEvents = await performQuery(`
        SELECT 
        e.id, e.line_id, l.line, e.shift_id, sh.shift_name, e.order_id, o.order_no, o.order_description, o.expected_qty, 
        o.product_id, p.product_code, p.product_name, p.speed AS product_speed,
        e.event_name, e.start_time, e.end_time, e.duration,
        e.speed, e.total_production, e.production_at_start, e.production_at_end,
        s.id AS stop_id, s.stop_start, s.main_stop_id, s.stop_end, s.downtime,
        s.downtime_reason_id, s.stop_category, s.stop_sub_category, s.stop_detail_level_1, 
        s.stop_detail_level_2, s.stop_detail_level_3, s.stop_detail_level_4, 
        s.equipment_id, s.sub_equipment_id, s.comment,	
        e.user_id, u.name
        FROM ${tables.events} e
        LEFT JOIN ${tables.line} l ON e.line_id = l.id
        LEFT JOIN ${tables.shift} sh ON e.shift_id = sh.id
        LEFT JOIN ${tables.orders} o ON e.order_id = o.id
        LEFT JOIN ${tables.product} p ON o.product_id = p.id
        LEFT JOIN ${tables.stops} s ON e.stop_id = s.id or (e.stop_id = s.main_stop_id and e.stop_id <> 0)
        LEFT JOIN ${tables.downtime_reasons} dr ON s.downtime_reason_id = dr.id
        LEFT JOIN ${tables.users} u ON e.user_id = u.id
        WHERE 
        STR_TO_DATE(e.start_time, '%Y-%m-%d %H:%i:%s') >= STR_TO_DATE('${apiResponse.start_date}', '%Y-%m-%d %H:%i:%s') 
        AND STR_TO_DATE(e.start_time, '%Y-%m-%d %H:%i:%s') < STR_TO_DATE('${apiResponse.end_date}', '%Y-%m-%d %H:%i:%s') 
        AND e.line_id = '${apiResponse.line_id}'
        ORDER BY 
        e.id DESC`
    );
    return fetchEvents;
    
  } catch (error) {
    console.log("error in fetching existing events: ", error);
  }
};



module.exports.dashboardCalculations = async (req, res) => {
  try {
    const { start_date, end_date, line_id } = req.body;

    const apiResponse = {
      start_date,
      end_date,
      line_id,
    };
    const orderDetails = await fetchOrderDetails(apiResponse);
    // const stops =  await fetchStops(apiResponse);
    const events = await fetchEvents(apiResponse);

    apiResponse.orderDetails = await transformData(orderDetails);
    apiResponse.events = events;

    // 1. Filter Events
    const runningEvents = events.filter(
      (event) => event.event_name === "Running"
    );
    const stopEvents = events.filter((event) => event.event_name !== "Running").map((event) => {
      return {
        id: event.stop_id,
        main_stop_id: event.main_stop_id,
        line_id: event.line_id,
        line: event.line,
        shift_id: event.shift_id,
        shift_name: event.shift_name,
        stop_start: event.stop_start,
        stop_end: event.stop_end,
        downtime: event.downtime,
        downtime_reason_id: event.downtime_reason_id,
        stop_category: event.stop_category,
        stop_sub_category: event.stop_sub_category,
        stop_detail_level_1: event.stop_detail_level_1,
        stop_detail_level_2: event.stop_detail_level_2,
        stop_detail_level_3: event.stop_detail_level_3,
        stop_detail_level_4: event.stop_detail_level_4,
        equipment_id: event.equipment_id,
        sub_equipment_id: event.sub_equipment_id,
        comment: event.comment,
      };
    });
    
    const unPlannedStopEvents = stopEvents.filter(
      (event) => event.stop_category === "Unplanned"
    );
    const plannedStopEvents = stopEvents.filter(
      (event) => event.stop_category === "Planned"
    );
    const utilizationStopEvents = stopEvents.filter(
      (event) => event.stop_category === "Utilization"
    );
    const unreportedStopEvents = stopEvents.filter(
      (event) => event.downtime_reason_id === 0
    );

    // 2. Calculate basic arrays (Scrap, Uptime, Downtime, etc.)
    const scrapArray = orderDetails.map((x) => safeNumber(x.scrap));
    const availableTimeArray = events.map((x) => safeNumber(x.duration));
    const upTimeArray = runningEvents.map((x) => safeNumber(x.duration));
    const downtimeArray = stopEvents.map((x) => safeNumber(x.downtime));
    const PDTArray = plannedStopEvents.map((x) => safeNumber(x.downtime));
    const UPDTArray = unPlannedStopEvents.map((x) => safeNumber(x.downtime));
    const UtilizationArray = utilizationStopEvents.map((x) =>
      safeNumber(x.downtime)
    );
    const UnreportedArray = unreportedStopEvents.map((x) =>
      safeNumber(x.downtime)
    );
    const totalOrderMinutesArray = events.map((x) =>
      x.main_stop_id == null || x.main_stop_id === 0
        ? safeNumber(x.duration)
        : 0
    );
    const weightedSpeedArray = runningEvents.map((x) =>
      isNaN(parseFloat(x.product_speed))
        ? 0
        : parseFloat(x.product_speed) * (safeNumber(x.duration) / 60)
    );

    // const expectedProductionArray = await orderDetails?.map((x) => safeNumber(x.expected_qty));
    const totalProductionArray = events?.map((x) =>
      x.main_stop_id == 0 || x.main_stop_id === null
        ? safeNumber(x.total_production)
        : 0
    );

    // 3. Date formatting and grouping (moved to helpers)
    const eventDataWithDate = events.map((e) => ({
      ...e,
      date: formatDate(e.start_time),
    }));
    const orderDetailWithDate = orderDetails.map((od) => ({
      ...od,
      date: formatDate(od.order_start),
    }));

    const orderDataAgainstProduct = groupBy(orderDetails, "product_code");
    const orderDataAgainstDate = groupBy(orderDetailWithDate, "date");
    const eventDataAgainstProduct = groupBy(events, "product_code");
    const eventDataAgainstDate = groupBy(eventDataWithDate, "date");

    // 4. OEE Calculations (using helper function)
    const calculatedByProduct = Object.entries(eventDataAgainstProduct).map(
      ([product_code, productEvents]) => {
        const orderDataForProduct = orderDataAgainstProduct[product_code];
        const oeeData = calculateOEE(productEvents, orderDataForProduct);
        return {
          x: product_code === "null" ? "No Active PO" : product_code,
          ...oeeData,
        };
      }
    );

    const calculatedByDate = Object.entries(eventDataAgainstDate).map(
      ([date, dateEvents]) => {
        const orderDataForDate = orderDataAgainstDate[date];
        const oeeData = calculateOEE(dateEvents, orderDataForDate);
        return {
          x: date === "null" ? "No Active PO" : date,
          ...oeeData,
        };
      }
    );
    // 5. Final Calculations (consolidated for efficiency)
    const scrap = scrapArray.reduce((a, b) => a + b, 0);
    const totalPDT = PDTArray.reduce((a, b) => a + b, 0) / 60;
    const totalUPDT = UPDTArray.reduce((a, b) => a + b, 0) / 60;
    const totalUtilization = UtilizationArray.reduce((a, b) => a + b, 0) / 60;
    const totalDowntime =
      downtimeArray.reduce((a, b) => a + b, 0) / 60 - totalUtilization;
    const totalOrderMinutes =
      totalOrderMinutesArray.reduce((a, b) => a + b, 0) / 60 - totalUtilization;
    const weightedSpeed = weightedSpeedArray.reduce((a, b) => a + b, 0);
    const availableTime = availableTimeArray.reduce((a, b) => a + b, 0) / 60;
    const totalUpTime = upTimeArray.reduce((a, b) => a + b, 0) / 60;
    const totalUnreported = UnreportedArray.reduce((a, b) => a + b, 0) / 60;
    // const plannedProduction = expectedProductionArray.reduce((a, b) => a + b, 0);

    const totalProduction = totalProductionArray.reduce((a, b) => a + b, 0);

    const goodProduction = totalProduction - scrap;

    const averageProductionSpeed =
      totalProduction === 0 ? 0 : totalProduction / totalUpTime;
    const avgProductSpeed =
      weightedSpeed === 0 || totalUpTime === 0
        ? 0
        : weightedSpeed / totalUpTime;
    const performance =
      avgProductSpeed === 0
        ? 0
        : (averageProductionSpeed / avgProductSpeed) * 100;
    const availability =
      totalUpTime === 0 || totalOrderMinutes === 0
        ? 0
        : (totalUpTime / totalOrderMinutes) * 100;
    const quality =
      goodProduction === 0 || totalProduction === 0
        ? 0
        : (goodProduction / totalProduction) * 100 > 100
        ? 100
        : (goodProduction / totalProduction) * 100;
    const getOEE =
      (availability / 100) * (performance / 100) * (quality / 100) * 100;

    // 6. Machine Stop Reasons (Simplified)
    const machineStopReason = stopEvents
      .filter((s) => s.stop_sub_category !== "Utilization")
      .reduce((acc, event) => {
        const stopSubCategory = event.stop_sub_category;
        acc[stopSubCategory] =
          (acc[stopSubCategory] || 0) + safeNumber(event.downtime) / 60;
        return acc;
      }, {});

    const stopsReasonTime = Object.entries(machineStopReason).sort(
      ([, timeA], [, timeB]) => timeB - timeA
    ); // Sort by downtime

    // 7. Frequent Stops (Simplified)
    const frequentStopsData = groupBy(stopEvents, "stop_sub_category");
    const frequentStops = Object.entries(frequentStopsData)
      .map(([type, stops]) => ({ type, value: stops.length }))
      .sort((a, b) => b.value - a.value);

    const uniqueOrderIds = new Set(events?.map((order) => order.order_id));
    const expectedProductionArray = events.reduce((acc, item) => {
      acc[item.order_id] = item.expected_qty;
      return acc;
    }, {});
    // console.log("expectedProductionArray: ", expectedProductionArray);

    const plannedProduction = Object.values(expectedProductionArray).reduce(
      (a, b) => a + b,
      0
    );

    // console.log("plannedProduction: ", plannedProduction);

    const MPS =
      plannedProduction == 0 && totalProduction == 0
        ? 0
        : Math.abs(plannedProduction - totalProduction) / plannedProduction;
    const mpsAdherence =
      plannedProduction == 0 && totalProduction == 0 ? 0 : (1 - MPS) * 100;

    const totalStops = downtimeArray?.length - UtilizationArray?.length || 0;
    const totalTime = moment(end_date).diff(start_date, "minutes");
    const shutdowntime = totalTime - availableTime;

    // Populate apiResponse with calculated values

    apiResponse.oeeDataByProduct = calculatedByProduct;
    apiResponse.oeeDataByDate = calculatedByDate;

    // Return available & Shutdowntime of Device
    apiResponse.totalTime = totalTime;
    apiResponse.availableTime = availableTime;
    apiResponse.shutdowntime = shutdowntime;

    // variables to use with stop tables 7 Charts
    apiResponse.stopEvents = stopEvents;
    apiResponse.machineStopReason = machineStopReason;
    apiResponse.stopsReasonTime = stopsReasonTime;
    apiResponse.frequentStops = frequentStops;
    apiResponse.uniqueOrderIds = Array.from(uniqueOrderIds);

    // Info for Availability, Performance, Quality, OEE with details
    apiResponse.availability = availability;
    apiResponse.operatingTime = totalUpTime;
    apiResponse.plannedProductionTime = totalOrderMinutes;
    apiResponse.nonProductionTime = totalUtilization;

    apiResponse.performance = performance;
    apiResponse.avgProductionSpeed = averageProductionSpeed;
    apiResponse.avgProductSpeed = avgProductSpeed;

    apiResponse.quality = quality;
    apiResponse.totalProduction = totalProduction;
    apiResponse.unit = orderDetails[0]?.qty_unit || "";
    apiResponse.goodProduction = goodProduction;
    apiResponse.scrap = scrap;

    apiResponse.oee = getOEE;

    // Downtime Info
    apiResponse.totalDowntime = totalDowntime;
    apiResponse.totalStops = totalStops;
    apiResponse.plannedDowntime = totalPDT;
    apiResponse.plannedStops = PDTArray?.length || 0;
    apiResponse.unplannedDowntime = totalUPDT;
    apiResponse.unplannedStops = UPDTArray?.length || 0;
    apiResponse.unreportedDowntime = totalUnreported;
    apiResponse.unreportedStops = UnreportedArray?.length || 0;

    // MPS Adherence
    apiResponse.mpsAdherence = mpsAdherence;
    apiResponse.mps = MPS * 100;
    apiResponse.totalProduction = totalProduction;
    apiResponse.plannedProduction = plannedProduction;

    // MTTR & MTBF
    apiResponse.mttr = totalStops === 0 ? 0 : totalDowntime / totalStops;
    apiResponse.mtbf = totalStops === 0 ? 0 : totalUpTime / totalStops;

    // Rewuired Fields in Response
    /*
            totalUniqueArea: plannedProduction,
            uniqueOrderIds: uniqueOrderIds,
            uniqueTufftingIds: uniqueTufftingIds,
            mpsAdherence: mpsAdherence,
            plannedProduction: plannedProduction,
            actualProduction: actualProduction,
            oeeDataByProduct: calculatedByProduct,
            oeeDataByDate: calculatedByDate,
            designedSpeed: weightedSpeed === 0 || totalUpTime === 0 ? 0 : weightedSpeed / totalUpTime,
            unplannedStopsCount: UPDTArray?.length || 0,
            plannedStopsCount: PDTArray?.length || 0,
            nonProductionStopsCount: NPTArray?.length || 0,
            utilizationHours: totalUtilization,
            utilizationStopsCount: UtilizationArray?.length || 0,
            unreportedHours: totalUnreported,
            unreportedStopsCount: UnreportedArray?.length || 0,
            nonProductionHours: totalNPT,
            plannedDownTime: totalPDT,
            unplannedDownTime: totalUPDT,
            stopEvents: stopEvents,
            performanceGauge: perfomanceGauge,
            avgSpeed: averageProductionSpeed,
            operationTime: totalUpTime,
            productCountTotal: totalProduction,
            shipableProducts: goodProduction,
            waste: wasteProduct,
            scrapCat: [scrapCat1, scrapCat2, scrapCat3],
            scrapPercent: (wasteProduct / totalProduction) * 100,
            plannedHours: totalOrderMinutes,
            totalUpTime: totalUpTime,
            totalHours: moment(endDate).diff(startDate, 'minutes'),

            oee: getOEE,
            performance: perfomance,
            quality: quality,
            availability: availability,

            availableHours: moment(endDate).diff(startDate, 'minutes'),
            totalDowntime: totalDowntime,
            totalStops: downtimeArray?.length - UtilizationArray?.length || 0,
            MTTR: totalDowntime / downtimeArray?.length,
            MTBF: totalUpTime / downtimeArray?.length,

            stopsReasonTime: stopsReasonTime,
            filterStops: stopEvents.filter(s => s.stop_category !== "Utilization"),
            frequentStops: frequentStops
        */

    return sendResponse(
      res,
      apiResponse,
      "Dashboard Calculations executed successfully",
      200
    );
  } catch (error) {
    console.log("Error in Dashboard Calculations: ", error);
    return sendErrorResponse(res, error, "Error in Dashboard Calculations");
  }
};




function transformData(rows) {
  const map = new Map();

  for (const row of rows) {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.order_id}-${row.line_id}-${row.product_code}`;

    if (!map.has(key)) {
      map.set(key, {
        id: row.order_id,
        line_id: row.line_id,
        line: row.line,
        product_id: row.product_id ?? null, // if exists in your real data
        product_code: row.product_code,
        product_name: row.product_name,
        product_speed: row.product_speed,
        expected_qty: row.expected_qty,
        qty_unit: row.qty_unit,
        order_no: row.order_no,
        order_description: row.order_description,
        order_status: row.order_status,
        production: 0,
        scrap: 0,
        shipable_product: 0,
        events: []
      });
    }

    const agg = map.get(key);

    // Sum production & scrap (and shipable_product if you want it aggregated too)
    agg.production += parseFloat(row.production) || 0;
    agg.scrap += parseFloat(row.scrap) || 0;
    agg.shipable_product += parseFloat(row.shipable_product) || 0;

    // Push event-level data
    agg.events.push({
      id: row.id,
      order_start: row.order_start,
      order_end: row.order_end,
      shift_id: row.shift_id,
      shift_name: row.shift_name
    });
  }

  return Array.from(map.values());
}
