const Piscina = require("piscina");
const path = require("path");
const { sendErrorResponse, sendResponse } = require("../utils");
// const socketHelper = require("../helper/socket.helper"); // Import socket.io setup
const { performQuery } = require("../utils/dbFunctions");
const tables = require("../utils/tables");
const { getShiftData } = require("../helper/shift.helper.js");
const { getSystemTime } = require("../functions/serverTime");
const moment = require("moment-timezone");
const { group } = require("console");

const piscina = new Piscina({
  filename: path.resolve(__dirname, "../multiThreading/worker.js"), // Path to worker file
  maxThreads: 60,
});

module.exports.createDeviceData = async (req, res) => {
  try {
    const {
      line_id, order_id,
      shift_id, downtime_reason_id,
      equipment_id, sub_equipment_id,
      total_production, status, 
      machine_speed, timestamp,
      vfd_1, vfd_2, vfd_3, vfd_4,
      vfd_5, vfd_6, vfd_7
    } = req.body;

    const deviceData = {
      line_id: Number(line_id),
      order_id: Number(order_id) || 0,
      shift_id: Number(shift_id),
      downtime_reason_id: Number(downtime_reason_id) || 0,
      equipment_id: Number(equipment_id) || 0,
      sub_equipment_id: Number(sub_equipment_id) || 0,
      total_production: parseFloat(total_production),
      status: Number(status),
      machine_speed: parseFloat(machine_speed),
      timestamp: timestamp,
      vfd_1: vfd_1,
      vfd_2: vfd_2,
      vfd_3: vfd_3,
      vfd_4: vfd_4,
      vfd_5: vfd_5,
      vfd_6: vfd_6,
      vfd_7: vfd_7,
    };
    

    // Delegate processing to the worker thread
    const ioData = await piscina.run(deviceData);
    console.log("ioData 1: ", ioData);

    // Get Socket.IO instance and emit data
    // const io = socketHelper.getIO();
    // io.emit("newDeviceData", ioData);

    // return sendResponse(res, ioData, "Success", 200);
    return sendResponse(res, ioData, "Success", 200);
  } catch (error) {
    console.log("Error inserting data: ", error);
    return sendErrorResponse(res, error, "Error inserting data");
  }
};

// This Query is Used by IoT Device to Get Current Order_id, Shift_id, Reset Counter nad Current PO Status (PO Active on Device Line)
module.exports.orderInfo = async (req, res) => {
  try {
    // Initialize API Response Object
    const apiResponse = {
      order_id: 0,
      shift_id: 0,
      order_no: '',
    };

    // Extract Data from Request Body
    const deviceData = {
      line_id: req.body.line_id,
      order_id: req.body.order_id || 0,
      shift_id: req.body.shift_id,
      timestamp: req.body.timestamp,
    };

    // Prepare Queries
    const query = {
      currentOrderQuery: `
        SELECT id, status, order_no 
        FROM ${tables.orders}
        WHERE line_id = "${deviceData.line_id}" and id="${deviceData.order_id}"`,

      activeOrderQuery: `
        SELECT id,status, order_no 
        FROM ${tables.orders} 
        WHERE line_id = "${deviceData.line_id}" and status = "1"`,
    };
    // Fetch Data from Database
    const currentPODetails = await performQuery(query.currentOrderQuery);
    const activePODetails = await performQuery(query.activeOrderQuery);
    const shiftData = await getShiftData(deviceData.timestamp);

    // print logs for debugging
    // console.log("currentPODetails: ", currentPODetails);
    // console.log("activePODetails: ", activePODetails);
    // console.log("shiftData: ", shiftData);

    // Prepare API Response
    apiResponse.order_id = activePODetails[0]?.id || 0;
    apiResponse.order_no = activePODetails[0]?.order_no;
    apiResponse.shift_id = shiftData[0]?.id;

    // console.log("API Response: ", apiResponse);
    

    // send Success Response
    return sendResponse(res, apiResponse, "Success", 200);

  } catch (error) {
    console.log("error in orderInfo: ", error);
    return sendErrorResponse(res, error, "Error getting order info");
  }
};

module.exports.getVfdData = async (req, res) => {
  try {
    // Get Filter Parameters from Request
    const { line_id, equipment_id, sub_equipment_id, start_date, end_date, product_id, is_realtime } = req.query;

    // Build SQL Query
    let query = ``;

    if (is_realtime && is_realtime == 1) {
      const device_data_id = await performQuery(`SELECT id FROM ${tables.device_data} ${line_id ? `WHERE line_id = "${line_id}"` : ''} ORDER BY id DESC LIMIT 1;`);
      if (device_data_id.length === 0) {
        console.log("device_data_id: ", device_data_id);
        return sendResponse(res, [], "VFD Data fetched successfully", 200);
      }
      query = `
        SELECT 
          vd.id, vd.line_id, l.line, se.equipment_id, 
          e.equipment, vd.sub_equipment_id, se.sub_equipment, 
          vd.order_id, o.product_id, p.product_name, p.product_code, 
          vd.timestamp, vd.vfd_speed, vd.vfd_voltage, vd.vfd_current
        FROM ${tables.vfd_data} vd
        LEFT JOIN ${tables.orders} o on vd.order_id = o.id
        LEFT JOIN ${tables.product} p on o.product_id = p.id
        LEFT JOIN ${tables.line} l on vd.line_id = l.id
        LEFT JOIN ${tables.sub_equipment} se on vd.sub_equipment_id = se.id
        LEFT JOIN ${tables.equipment} e on se.equipment_id = e.id
        WHERE vd.device_data_id = "${device_data_id[0].id}"
        ORDER BY vd.timestamp ASC;
      `
    }
    else {
      console.log(`Fetching VFD data for line_id: ${line_id}, sub_equipment_id: ${sub_equipment_id}, start_date: ${start_date}, end_date: ${end_date}`);
      
      if (!start_date || !end_date || !line_id || !sub_equipment_id) {
        return sendErrorResponse(res, null, "start_date and end_date are required");
      }
      // Calculate Time Difference and Determine Grouping Interval
      const start = moment(start_date);
      const end = moment(end_date);
      const diffInMinutes = end.diff(start, 'minutes');
      console.log(`Time difference in minutes: ${diffInMinutes}`);
      

      const recordsPerMinute = 3; // because data geather at 1 record every 20s
      const totalRecords = diffInMinutes * recordsPerMinute;

      const maxRecordsThreshold = 1200;        // your max ~points for chart
      const groupingFactor = totalRecords / maxRecordsThreshold;

      // groupingTime in *minutes* per bucket (like before)
      const groupingTimeMinutes =
        groupingFactor > 1 ? groupingFactor / recordsPerMinute : 1;

      // convert to seconds
      let rawIntervalSeconds = groupingTimeMinutes * 60;

      // 🔹 base step is 20 seconds – snap interval to a multiple of 20s
      const baseStep = 20;
      let intervalSeconds = Math.ceil(rawIntervalSeconds / baseStep) * baseStep;

      // avoid 0 or < 20s
      if (intervalSeconds < baseStep) {
        intervalSeconds = baseStep;
      }

      console.log('intervalSeconds (bucket size):', intervalSeconds);

      // (optional) only for logging in H/M/S form
      const groupHour = Math.floor(intervalSeconds / 3600);
      const groupMinutes = Math.floor((intervalSeconds % 3600) / 60);
      const groupSeconds = intervalSeconds % 60;

      console.log(
        `Grouping Time - Hours: ${groupHour}, Minutes: ${groupMinutes}, Seconds: ${groupSeconds}`
      );

      query = `
        SELECT 
          vd.line_id,
          l.line,
          se.equipment_id,
          e.equipment,
          vd.sub_equipment_id,
          se.sub_equipment,

          -- formatted bucket as timestamp
          DATE_FORMAT(
            FROM_UNIXTIME(
              FLOOR(
                UNIX_TIMESTAMP(STR_TO_DATE(vd.timestamp, '%Y-%m-%d %H:%i:%s'))
                / ${intervalSeconds}
              ) * ${intervalSeconds}
            ),
            '%Y-%m-%d %H:%i:%s'
          ) AS bucket_timestamp,

          -- aggregated values for this bucket
          ROUND(AVG(vd.vfd_speed),2)   AS vfd_speed,
          ROUND(AVG(vd.vfd_voltage),2) AS vfd_voltage,
          ROUND(AVG(vd.vfd_current),2) AS vfd_current

        FROM ${tables.vfd_data} vd
        LEFT JOIN ${tables.orders} o         ON vd.order_id         = o.id
        LEFT JOIN ${tables.product} p        ON o.product_id        = p.id
        LEFT JOIN ${tables.line} l           ON vd.line_id          = l.id
        LEFT JOIN ${tables.sub_equipment} se ON vd.sub_equipment_id = se.id
        LEFT JOIN ${tables.equipment} e      ON se.equipment_id     = e.id

        WHERE 
          STR_TO_DATE(vd.timestamp, '%Y-%m-%d %H:%i:%s')
            BETWEEN STR_TO_DATE('${start_date}', '%Y-%m-%d %H:%i:%s')
            AND     STR_TO_DATE('${end_date}',   '%Y-%m-%d %H:%i:%s')
          AND vd.line_id = "${line_id}"
          AND vd.sub_equipment_id = "${sub_equipment_id}"
          ${equipment_id ? `AND se.equipment_id = "${equipment_id}"` : ''}
          ${product_id   ? `AND o.product_id   = "${product_id}"`   : ''}

        GROUP BY
          bucket_timestamp,
          vd.line_id,
          vd.sub_equipment_id

        ORDER BY bucket_timestamp ASC;
      `;

    }

    console.log("query: ", query);
    
    // Execute Query
    const data = await performQuery(query);
    console.log("Data Length: ", data?.length);

    // Return Response
    return sendResponse(res, data, "VFD Data fetched successfully", 200);

  } catch (error) {
    console.log("Error in Get VFD Data: ", error);
    return sendErrorResponse(res, error, "Error getting VFD data");
  }
}