const moment = require("moment-timezone");
const { performQuery } = require("../utils/dbFunctions");
const { sendErrorResponse, sendResponse } = require("../utils");
const tables = require("../utils/tables");
const { getSystemTime, getTimezone } = require("../functions/serverTime");
const { getShiftData } = require("../helper/shift.helper");

module.exports.getTemperature = async (req, res) => {
  try {
    //==> Get system time and timezone
    const systemTime = await getSystemTime();
    const timezone = await getTimezone();
    const timestamp = moment(systemTime)
      .tz(timezone)
      .format("YYYY-MM-DD HH:mm:ss");

    // console.log("systemTime => : ", systemTime);
    // console.log("timestamp => : ", timestamp);

    //==> Get the shift data based on the current timestamp
    const shiftData = await getShiftData(timestamp);
    // console.log("shiftData: ", shiftData);

    //==> Check if shiftData exists
    if (shiftData.length === 0) {
      return sendResponse(res, [], "No active shift found", 200);
    }

    // Extract the shift name from shiftData
    const shiftName = shiftData[0].shiftName;

    const query = `
    SELECT *
    FROM ${tables.device_data}
    WHERE shift_name = ? 
      AND DATE(STR_TO_DATE(timestamp, '%m/%d/%Y %H:%i:%s')) = CURDATE();
  `;

    // console.log("Query: ", query);
    const data = await performQuery(query, [shiftName]);

    //==> Check if data not exists
    if (data.length === 0) {
      return sendResponse(
        res,
        [],
        "No Temperature Data Found for the Current Shift",
        200
      );
    }

    // console.log("Data => ", data);
    return sendResponse(
      res,
      data,
      "Temperature data fetched successfully",
      200
    );
  } catch (error) {
    console.error("Error fetching temperature data:", error);
    return sendErrorResponse(res, error, "Error fetching temperature data");
  }
};
