const { performQuery } = require("../utils/dbFunctions");
const { sendErrorResponse, sendResponse } = require("../utils");
const tables = require("../utils/tables");
const { getSystemTime } = require("../functions/serverTime");
const moment = require("moment-timezone");
const { isUndefined } = require("lodash");

//=====> GET TUFTING TICKETS
module.exports.getFilteredTuftingTickets = async (req, res) => {
  try {
    const { startDate, endDate, line_name } = req.query;
    const query = `
      SELECT 
        tt.id,
        o.product_id As product_id, p.product, 
        p.product_name, o.order_id, 
        tt.trolly_no, tt.status,
        tt.width, tt.roll_length, tt.operator,
        tt.pile_height, tt.no_of_rolls, tt.completed_rolls,
        tt.roll_size, tt.total_area, tt.sales_order, 
        tt.trolly_carpet_width, l.line,
        o.created_at, o.updated_at, 
        uC.name AS created_by, uU.name AS updated_by,
         o.line_id AS line_id
      FROM ${tables.tufting_ticket} AS tt
      LEFT JOIN 
        ${tables.orders} AS o 
        ON o.id = tt.order_id
      LEFT JOIN 
        ${tables.line} AS l 
        ON l.id = tt.line_id
      LEFT JOIN 
        ${tables.product} AS p 
        ON p.id = o.product_id
      LEFT JOIN 
        ${tables.users} AS uC 
        ON uC.id = o.created_by
      LEFT JOIN 
        ${tables.users} AS uU 
        ON uU.id = o.updated_by
    WHERE 
      (
        (
          STR_TO_DATE(tt.updated_at, '%m-%d-%Y %H:%i:%s') >= STR_TO_DATE('${startDate}', '%m/%d/%Y %H:%i:%s') 
          AND STR_TO_DATE(tt.updated_at, '%m-%d-%Y %H:%i:%s') < STR_TO_DATE('${endDate}', '%m/%d/%Y %H:%i:%s') 
        )
          OR (tt.status <> 2 AND tt.status <> 3) 
      )
      ${line_name ? `AND l.line = '${line_name}'` : ""}
      ORDER BY tt.id DESC;
    `;

    console.log(query);
    
    const data = await performQuery(query);
    if (data.length === 0) {
      return sendResponse(res, data, "No Data Found", 200);
    }
    return sendResponse(res, data, "selected Successfully", 200);
  } catch (error) {
    console.log("error: ", error);

    return sendErrorResponse(res, error, "Error getting tuftingTicket API");
  }
};

//=====> GET TUFTING TICKETS
module.exports.getTuftingTickets = async (req, res) => {
  try {
    const query = `
      SELECT 
        tt.id,
        o.product_id As product_id, p.product, 
        p.product_name, o.order_id, 
        tt.trolly_no, tt.status,
        tt.width, tt.roll_length, tt.operator,
        tt.pile_height, tt.no_of_rolls, tt.completed_rolls,
        tt.roll_size, tt.total_area, tt.sales_order, 
        tt.trolly_carpet_width,
        o.created_at, o.updated_at, 
        uC.name AS created_by, uU.name AS updated_by,
         o.line_id AS line_id
      FROM ${tables.tufting_ticket} AS tt
      LEFT JOIN 
        ${tables.orders} AS o 
        ON o.id = tt.order_id
      LEFT JOIN 
        ${tables.product} AS p 
        ON p.id = o.product_id
      LEFT JOIN 
        ${tables.users} AS uC 
        ON uC.id = o.created_by
      LEFT JOIN 
        ${tables.users} AS uU 
        ON uU.id = o.updated_by
      ORDER BY tt.id DESC;
    `;
    const data = await performQuery(query);
    if (data.length === 0) {
      return sendResponse(res, data, "No Data Found", 200);
    }
    return sendResponse(res, data, "selected Successfully", 200);
  } catch (error) {
    console.log("error: ", error);

    return sendErrorResponse(res, error, "Error getting tuftingTicket API");
  }
};

//=====> CREATE TUFTING TICKET
module.exports.createTuftingTicket = async (req, res) => {
  try {
    let {
      line_id,
      product_id,
      order_id,
      trolly_no,
      roll_length,
      roll_size,
      operator,
      width,
      pile_height,
      sales_order,
      no_of_rolls,
      total_area,
      user_id,
      trolly_carpet_width
      // status
    } = req.body;

    //====> Validation
    if (
      !line_id ||
      !product_id ||
      !order_id ||
      !trolly_no ||
      !roll_length ||
      !roll_size ||
      !operator ||
      !width ||
      !sales_order ||
      !no_of_rolls ||
      !total_area ||
      !user_id
    ) {
      return sendErrorResponse(res, 400, "Required Information Missing", 400);
    }

    const order_id_array = order_id.split("GVCT0")
    const trolly_no_array = trolly_no.split("-")
    const sales_order_array = sales_order.split("GVSO")    

    order_id = "GVCT0"+order_id_array[order_id_array.length-1]
    trolly_no = order_id+"-"+trolly_no_array[trolly_no_array.length-1]
    sales_order = "GVSO"+sales_order_array[sales_order_array.length-1]

    
    console.log("data: ", {    
      line_id: line_id,
      product_id: product_id,
      order_id: order_id,
      trolly_no: trolly_no,
      roll_length: roll_length,
      roll_size: roll_size,
      operator: operator,
      width: width,
      pile_height: pile_height,
      sales_order: sales_order,
      no_of_rolls: no_of_rolls,
      total_area: total_area,
      user_id: user_id,
      trolly_carpet_width: trolly_carpet_width})



    // check for already existing Tufting Trolley
    const existingTrolly = await performQuery(`Select * From ${tables.tufting_ticket} Where trolly_no = ?`,[trolly_no])
    if (existingTrolly.length > 0) {
      return sendErrorResponse(res, 409, "Trolley No Already Exist", 409);
    }

    //  Check for Existing Order
    let id_of_order;
    const existingOrderQuery = `SELECT * FROM ${tables.orders} WHERE order_id = ? and 	line_id = ?`;
    const existingOrderData = await performQuery(existingOrderQuery, [
      order_id,
      line_id,
    ]);
    let orderData;
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    if (existingOrderData.length > 0) {
      id_of_order = existingOrderData[0].id;
      // Update existing order
      const orderUpdateQuery = `UPDATE ${tables.orders} SET ? WHERE id = ?`;
      const orderUpdateParams = [
        {
          expected_qty:
            parseFloat(existingOrderData[0].expected_qty) +
            parseFloat(total_area),
        },
        existingOrderData[0].id,
      ];
      orderData = await performQuery(orderUpdateQuery, orderUpdateParams);
    } else {
      // create New Order
      const orderInsertQuery = `INSERT INTO ${tables.orders} SET ?`;
      const orderInsertParams = [
        {
          order_id: order_id,
          line_id: line_id,
          expected_qty: total_area,
          qty_unit: "Sq.Mtr",
          product_id: product_id,
          expected_start_date: moment(systemTime).format("YYYY-MM-DD"),
          created_at: currentTime,
          updated_at: currentTime,
          created_by: user_id,
          updated_by: user_id,
        },
      ];
      orderData = await performQuery(orderInsertQuery, orderInsertParams);

      id_of_order = orderData.insertId;
    }

    console.log("orderData: ", orderData);

    // Create Tufting Entry against Id from Order Table
    const tuftingInsertQuery = `INSERT INTO ${tables.tufting_ticket} SET ?`;
    const tuftingInsertParams = [
      {
        order_id: id_of_order,
        line_id: line_id,
        trolly_no: trolly_no,
        roll_length: roll_length,
        roll_size: roll_size,
        operator: operator,
        width: width,
        no_of_rolls: no_of_rolls,
        pile_height: pile_height || 0,
        sales_order: sales_order,
        total_area: total_area,
        created_at: currentTime,
        created_by: user_id,
        updated_at: currentTime,
        updated_by: user_id,
        trolly_carpet_width: trolly_carpet_width || 0,
      },
    ];
    const tuftingData = await performQuery(
      tuftingInsertQuery,
      tuftingInsertParams
    );

    if (tuftingData.insertId > 0) {
      return sendResponse(res, tuftingData, "created successfully", 201);
    }
    return sendErrorResponse(res, 400, "Error in Creating Tufting Data", 400);
  } catch (error) {
    console.log("error: ", error);

    return sendErrorResponse(res, error, "Error in Create Defect Reason API");
  }
};

//=====> UPDATE TUFTING TICKET
module.exports.updateTuftingTicket = async (req, res) => {
  try {
    let {
      id,
      line_id,
      product_id,
      order_id,
      trolly_no,
      roll_length,
      roll_size,
      operator,
      width,
      pile_height,
      sales_order,
      no_of_rolls,
      total_area,
      user_id,
      trolly_carpet_width,
      status,
    } = req.body;

    
    const order_id_array = order_id.split("GVCT0")
    const trolly_no_array = trolly_no.split("-")
    // const sales_order_array = sales_order.split("GVSO")    

    order_id = "GVCT0"+order_id_array[order_id_array.length-1]
    trolly_no = order_id+"-"+trolly_no_array[trolly_no_array.length-1]
    // sales_order = "GVSO"+sales_order_array[sales_order_array.length-1]``

    //====> Validation
    if (
      !id ||
      !line_id ||
      !product_id ||
      !order_id ||
      !trolly_no ||
      !roll_length ||
      !roll_size ||
      !operator ||
      !width ||
      // !sales_order ||
      !no_of_rolls ||
      !total_area ||
      !user_id ||
      status === undefined
      // isUndefined(status)
    ) {
      return sendErrorResponse(res, 400, "Required Information Missing", 400);
    }

    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Fetch Existing Tufting Ticket
    const existingTTQuery = `SELECT * FROM ${tables.tufting_ticket} WHERE id = ?`;
    const existingTTData = await performQuery(existingTTQuery, [id]);
    // console.log("existingTTData => ", existingTTData);
    if (existingTTData.length == 0) {
      return sendErrorResponse(res, 404, "Tufting Ticket Not Found", 404);
    }

    //  Fetch Existing Order
    const existingOrderQuery = `SELECT * FROM ${tables.orders} WHERE order_id = ? and line_id = ?`;
    const existingOrderData = await performQuery(existingOrderQuery, [
      order_id,
      line_id,
    ]);
    if (existingOrderData.length == 0) {
      return sendErrorResponse(
        res,
        404,
        "Order Against Tufting Ticket Not Found",
        404
      );
    }
    
    // Check for Changes
    const changeInTrolly_no = existingTTData[0].trolly_no !== trolly_no;
    const changeInWidth = existingTTData[0].width !== width;
    const changeInRoll_length = existingTTData[0].roll_length !== roll_length;
    const changeInRoll_size = existingTTData[0].roll_size !== roll_size;
    const changeInOperator = existingTTData[0].operator !== operator;
    const changeInPile_height = existingTTData[0].pile_height !== pile_height;
    const changeInStatus = existingTTData[0].status !== status;

    if (
      !changeInTrolly_no &&
      !changeInWidth &&
      !changeInRoll_length &&
      !changeInRoll_size &&
      !changeInOperator &&
      !changeInPile_height &&
      !changeInStatus
    ) {
      return sendErrorResponse(res, 400, "No Changes Detected", 400);
    }

    
    
    
    const runningOrderDetail = await performQuery(`Select * from ${tables.order_detail} Where tufting_id = ?`, [id])
    console.log("runningOrderDetail", runningOrderDetail);
    

    
    //===> Checking active tufting ticket
    const checkforActiveTT = await performQuery(
      `SELECT tt.id, tt.line_id, tt.order_id, tt.status, o.order_id AS poNumber
      FROM ${tables.tufting_ticket} AS tt 
      Left JOIN ${tables.orders} AS o on tt.order_id = o.id
      WHERE o.order_id = ? and tt.line_id = ? and (tt.status = 1 OR tt.status = 4) and tt.id <> ? `, 
      [order_id, line_id, id]
    );

    // Check for any other Active tufting ticket on same line
    if(changeInStatus && status == 1){
      const otherActiveTT = await performQuery(`Select * From ${tables.tufting_ticket} where id<>? and status =1 and line_id = ? `, [id, line_id])
      if(otherActiveTT.length > 0){
        return sendErrorResponse(res, 400, "Cannot Update Tufting Ticket: Another Active Ticket Exists", 400);
      }
    }

    let updateTTQuery
    let updateTTParams
    let updateOrderQuery
    let updateOrderParams

    if (existingTTData[0].status == 0) {
      // User is Allowed to Update all Editable Fields
      updateTTQuery = `UPDATE ${tables.tufting_ticket} SET ? WHERE id = ?`;
      updateTTParams = [
        {
          trolly_no: trolly_no,
          width: width,
          roll_length: roll_length,
          roll_size: roll_size,
          operator: operator,
          pile_height: pile_height || 0,
          no_of_rolls: Number(roll_length / roll_size),
          total_area: parseFloat(roll_length * width),
          status: status,
          updated_at: currentTime,
          updated_by: user_id,
          trolly_carpet_width: trolly_carpet_width,
        },
        id,
      ];
      updateOrderQuery = `UPDATE ${tables.orders} SET ? WHERE order_id = ?`;
      updateOrderParams = [
        {
          expected_qty:
            parseFloat(existingOrderData[0].expected_qty) -
            parseFloat(existingTTData[0].total_area) +
            parseFloat(roll_length * width),
          status: status != 2 ? status : 3,
          updated_at: currentTime,
          updated_by: user_id,
        },
        order_id,
      ];
    } else if (existingTTData[0].status != 2) {
      // User is Allowed to Update Status Only
      // If the status is 1 (Active) or 3 (Hold) then only the status field can be updated, but not the other editable fields
      if (!changeInStatus) {
        return sendErrorResponse(res, 400, "No Changes Detected", 400);
      } else {
        // User is Allowed to Update Status Only
        updateTTQuery = `UPDATE ${tables.tufting_ticket} SET ? WHERE id = ?`;
        updateTTParams = [
          {
            status: status,
            updated_at: currentTime,
            updated_by: user_id,
          },
          id,
        ];
        updateOrderQuery = `UPDATE ${tables.orders} SET ? WHERE order_id = ?`;
        updateOrderParams = [
          {
            status: status != 2 ? status : 3,
            updated_at: currentTime,
            updated_by: user_id,
          },
          order_id,
        ];


      }
    } else {
      // Since the status is 2 (Completed) no changes are allowed to be made
      return sendErrorResponse(res, 403, "Unauthorized to Update Tufting Ticket", 403);
    }

    // console.log("checkforActiveTT: ", checkforActiveTT);    
    
    const ttData = await performQuery(updateTTQuery, updateTTParams);

    //===> Checking status = 2 (Completed) then update scrap in order_details
    if (status == 2) {
      const sumRollLengthQuery = `
      SELECT COALESCE(SUM(roll_length), 0) AS total_finished_length
      FROM ${tables.finish_product_detail}
      WHERE tufting_ticket_id = ?`;

      const sumRollLengthData = await performQuery(sumRollLengthQuery, [id]);
      // console.log("sumRollLengthData => ", sumRollLengthData);

      const totalFinishedLength = sumRollLengthData[0].total_finished_length;
      const remainingRollLength = runningOrderDetail?.[0]?.product_length - totalFinishedLength;

      // const scrap = remainingRollLength

      console.log("remainingRollLength => ", remainingRollLength);

      //===> Insert remainingRollLength in order_details (scrap) field
      const scrap = `UPDATE ${tables.order_detail} SET ? Where id = ?`;
      const scrapParams = [{
        scrap: remainingRollLength, 
        tufting_id: id},
        runningOrderDetail[0].id
      ];
      await performQuery(scrap, scrapParams);
      // const scrapParams = [
      //   {
      //     scrap: remainingRollLength,
      //   },
      //   id,
      // ];
      // await performQuery(scrap, scrapParams);
    }

    let orderData = checkforActiveTT?.length == 0 ? (await performQuery(updateOrderQuery, updateOrderParams)) : null;
    let updateTTData = { ttData: ttData, orderData: orderData };

    if (ttData.affectedRows > 0) {
      if(existingTTData[0].status == 0){
        return sendResponse(res, updateTTData, "updated successfully", 200);
      }
      else{
        return sendResponse(res, updateTTData, "Only status updated successfully", 200);
      }
    }
    return sendErrorResponse(res, 400, "Error in Updating Tufting Data", 400);


  } catch (error) {
    console.log("error => ", error);
    return sendErrorResponse(
      res,
      500,
      error.message || "Error updating Tufting tickets"
    );
  }
};

//=====> DELETE TUFTING TICKET
module.exports.deleteTuftingTicket = async (req, res) => {
  try {
    const { id } = req.params;

    const query = `DELETE FROM ${tables.tufting_ticket} WHERE id=?`;
    const data = await performQuery(query, [id]);

    if (data.affectedRows > 0) {
      return sendResponse(res, data, "Deleted Successfully", 200);
    }

    return sendErrorResponse(
      res,
      400,
      "Error in Deleting Tufting tickets",
      400
    );
  } catch (error) {
    return sendErrorResponse(res, error, "Error deleting Tufting tickets");
  }
};
