const { parentPort } = require("worker_threads");
const moment = require("moment");
const { performQuery } = require("../utils/dbFunctions");
const { getShiftTimes, getShiftData } = require("../helper/shift.helper");
const tables = require("../utils/tables");
const { getSystemTime } = require("../functions/serverTime");

let apiResponse = {};

// Helper Function to get current Stop ID
const getCurrentStopId = async (apiResponse) => {
  // if device condition is Running then Stop ID will be 0
  if (apiResponse.device_condition === "Running") {
    return 0;
  }
  else{
    return apiResponse.current_stop?.id || 0;
  }
};

const getTimestamp = (timestamp) => {
  const time = moment(timestamp, "MM/DD/YYYY HH:mm:ss");
  if (time.hour() < 7) {
    time.subtract(1, "days");
  }
  return time;
};

const fetchAttendanceData = async (deviceData) => {
  const selectAttendance = `
    SELECT * FROM ${tables.attendance} WHERE
    line_id=? AND 
    STR_TO_DATE(punch_in, '%Y-%m-%d %H:%i:%s') <= STR_TO_DATE(?, '%Y-%m-%d %H:%i:%s') AND 
    (
    STR_TO_DATE(punch_out, '%Y-%m-%d %H:%i:%s') >= STR_TO_DATE(?, '%Y-%m-%d %H:%i:%s') OR 
    punch_out IS NULL
    ) 
    `;
  const attendanceQueryParams = [deviceData.line_id, deviceData.timestamp, deviceData.timestamp];
  
  return await performQuery(selectAttendance, attendanceQueryParams);
};

const insertDeviceData = async (deviceData, currentTime) => {
  const device_data = await performQuery(
    `INSERT INTO ${tables.device_data} SET ?`,{
      line_id: deviceData.line_id,
      order_id: deviceData.order_id,
      shift_id: deviceData.shift_id,
      downtime_reason_id: deviceData.downtime_reason_id,
      equipment_id: deviceData.equipment_id,
      sub_equipment_id: deviceData.sub_equipment_id,
      total_production: deviceData.total_production,
      status: deviceData.status,
      machine_speed: deviceData.machine_speed,
      timestamp: deviceData.timestamp,
      created_at: currentTime,
    }
  );
  
  const query = `
    INSERT INTO ${tables.vfd_data} 
    (device_data_id, line_id, shift_id, order_id, sub_equipment_id, vfd_voltage, vfd_current, vfd_speed, timestamp, created_at) VALUES
    (?, ?, ?, ?, ?, ?, ?, ?, ?, ?), (?, ?, ?, ?, ?, ?, ?, ?, ?, ?), (?, ?, ?, ?, ?, ?, ?, ?, ?, ?), (?, ?, ?, ?, ?, ?, ?, ?, ?, ?), 
    (?, ?, ?, ?, ?, ?, ?, ?, ?, ?), (?, ?, ?, ?, ?, ?, ?, ?, ?, ?), (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
  `;
  
  const params = [
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_1.id, deviceData.vfd_1.voltage, deviceData.vfd_1.current, deviceData.vfd_1.speed, deviceData.timestamp, currentTime,
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_2.id, deviceData.vfd_2.voltage, deviceData.vfd_2.current, deviceData.vfd_2.speed, deviceData.timestamp, currentTime,
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_3.id, deviceData.vfd_3.voltage, deviceData.vfd_3.current, deviceData.vfd_3.speed, deviceData.timestamp, currentTime,
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_4.id, deviceData.vfd_4.voltage, deviceData.vfd_4.current, deviceData.vfd_4.speed, deviceData.timestamp, currentTime,
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_5.id, deviceData.vfd_5.voltage, deviceData.vfd_5.current, deviceData.vfd_5.speed, deviceData.timestamp, currentTime,
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_6.id, deviceData.vfd_6.voltage, deviceData.vfd_6.current, deviceData.vfd_6.speed, deviceData.timestamp, currentTime,
    device_data.insertId, deviceData.line_id, deviceData.shift_id, deviceData.order_id, deviceData.vfd_7.id, deviceData.vfd_7.voltage, deviceData.vfd_7.current, deviceData.vfd_7.speed, deviceData.timestamp, currentTime,
  ]

  const vfd_data = await performQuery(query, params);

  return device_data;
};

const fetchActiveOrder = async (deviceData) => {
  try {
    const activeOrders = await performQuery(`
      SELECT id, status, order_no 
      FROM ${tables.orders}
      WHERE line_id = "${deviceData.line_id}" and status = 1`);
    return activeOrders;
  } catch (error) {
    console.log("error in fetching Active Orders: ", error);
  }
};

const fetchOrderDetails = async (deviceData, apiResponse) => {
  try {
    const fetchorderDetails = await performQuery(`
      SELECT * FROM ${tables.order_details} where (
      line_id = ? AND
      shift_id = ? AND
      order_id = ? AND
      status = 'In Progress' AND
        (
            STR_TO_DATE(order_start,'%Y-%m-%d %H:%i:%s') >= STR_TO_DATE(?,'%Y-%m-%d %H:%i:%s') AND
            STR_TO_DATE(order_start,'%Y-%m-%d %H:%i:%s') <= STR_TO_DATE(?,'%Y-%m-%d %H:%i:%s')
        )
      ) ORDER BY id DESC`,
      [
        deviceData.line_id,
        apiResponse.current_shift.id,
        deviceData.order_id,
        apiResponse.current_shift.shift_start,
        apiResponse.current_shift.shift_end,
      ]
    );

    return fetchorderDetails;
  } catch (error) {
    console.log("error in fetching order details: ", error);
  }
};

const fetchExistingStop = async (deviceData, apiResponse) => {
  try {
    const fetchStops = await performQuery(`
      SELECT * FROM ${tables.stops} 
      where (
        line_id = ? AND
        shift_id = ? AND
        order_id = ? AND
        user_id = ? AND
        status = 'In Progress'
      ) 
      ORDER BY id DESC`,
      [
        deviceData.line_id,
        apiResponse.current_shift.id,
        deviceData.order_id,
        apiResponse.current_user_id,
      ]
    );
  
    return fetchStops;
    
  } catch (error) {
    console.log("error in fetching existing stops: ", error);
  }
};

const fetchExistingEvent = async (deviceData, apiResponse) => {
  try {
    const fetchEvents = await performQuery(`
          SELECT * FROM ${tables.events} 
          WHERE (
              line_id = ? AND
              shift_id = ? AND
              order_id = ? AND 
              user_id = ? AND
              event_name = ? AND
              status = "In Progress" 
          )
          ORDER BY id DESC`,
          [
            deviceData.line_id,
            apiResponse.current_shift.id,
            deviceData.order_id,
            apiResponse.current_user_id,
            apiResponse.device_condition,
          ]
        );
    return fetchEvents;
    
  } catch (error) {
    console.log("error in fetching existing events: ", error);
  }
};

const updateOrderDetails = async (deviceData, apiResponse) => {
  try {
    const updateResult = await performQuery(
      `UPDATE ${tables.order_details} SET ? WHERE id= ?`,[{
        production: deviceData.total_production,
        machine_speed: deviceData.machine_speed,
        order_end: deviceData.timestamp,
        updated_at: apiResponse.current_timestamp,
      }, apiResponse.order_details.id]
    );
    return updateResult;
    
  } catch (error) {
    console.log("error in updating order details: ", error);
  }
};

const insertOrderDetails = async (deviceData, apiResponse) => {
  try {
    const InsertResult =  await performQuery(
      `INSERT INTO ${tables.order_details} SET ?`, {
        line_id: deviceData.line_id,
        order_id: deviceData.order_id,
        shift_id: apiResponse.current_shift.id,
        team_id: 0,
        status: "In Progress",
        order_start: deviceData.timestamp,
        order_end: deviceData.timestamp,
        production: deviceData.total_production,
        machine_speed: deviceData.machine_speed,
        shipable_product: deviceData.total_production,
        scrap: 0,
        created_at: apiResponse.current_timestamp,
        updated_at: apiResponse.current_timestamp,
      }
    );

    return InsertResult;

  } catch (error) {
    console.log("error in inserting order details: ", error);
  }
};

const closePreviousOrderDetails = async (deviceData, apiResponse) => {
  try {
    return await performQuery(
      `UPDATE ${tables.order_details} SET ? WHERE (line_id = ? AND id <> ? AND status = ?)`,
      [{
        status: 'Completed',
        updated_at: apiResponse.current_timestamp
      }, 
        deviceData.line_id, 
        apiResponse.new_order_details.id,
        'In Progress'
      ]
    );
    
  } catch (error) {
    console.log("error in closing previous order details: ", error);
  }
};

const createNewStop = async (deviceData, apiResponse) => {
  try {
    const insertStop = await performQuery(
      `INSERT INTO ${tables.stops} SET ?`, {
        main_stop_id: 0,
        line_id: deviceData.line_id,
        shift_id: apiResponse.current_shift.id,
        order_id: deviceData.order_id,
        user_id: apiResponse.current_user_id,
        device_data_id: apiResponse.device_data_id,
        stop_start: deviceData.timestamp,
        stop_end: deviceData.timestamp,
        downtime: 0,
        order_detail_id: apiResponse.current_order_details.id,
        stop_category: "Unreported",
        stop_sub_category: "Unreported",
        status: "In Progress",
        created_at: apiResponse.current_timestamp,
        updated_at: apiResponse.current_timestamp,
      }
    );
    
    return insertStop;
    
  } catch (error) {
    console.log("error in creating new stop: ", error);
  }
};

const updateStop = async (deviceData, apiResponse) => {
  try {
    const downtime =
      (moment.utc(deviceData.timestamp) - moment.utc(apiResponse.current_stop.stop_start)) / 1000;
    
    const updateStop = await performQuery(
      `UPDATE ${tables.stops} SET ? WHERE id= ?`, [{
        stop_end: deviceData.timestamp,
        downtime: downtime,
        updated_at: apiResponse.current_timestamp,
      }, apiResponse.current_stop.id]
    );
    
    return updateStop;
    
  } catch (error) {
    console.log("error in updating stop: ", error);
  }
};

const closeStop = async (deviceData, stop_data) => {
  try {
    const previousOpenStops = await performQuery(
      `SELECT * FROM ${tables.stops} 
      WHERE (line_id = ? AND id <> ? AND status = ?)`,
      [deviceData.line_id, stop_data.id, "In Progress"]
    );

    if (previousOpenStops.length > 0) {
      previousStop = previousOpenStops[0];
      const downtime =
        (moment.utc(deviceData.timestamp) - moment.utc(previousStop.stop_start)) / 1000;
  
      const updateStop = await performQuery(
        `UPDATE ${tables.stops} SET ? WHERE id = ?`, [{
          stop_end: deviceData.timestamp,
          downtime: downtime,
          status: "Completed",
          updated_at: apiResponse.current_timestamp,
        }, previousStop.id]
      );
    }

    
    return true;
    
  } catch (error) {
    console.log("error in closing stop: ", error);
  }
};

const closeSpecificStop = async (deviceData, stop_data) => {
  try {
    const downtime =
      (moment.utc(deviceData.timestamp) - moment.utc(stop_data.stop_start)) / 1000;

    const updateStop = await performQuery(
      `UPDATE ${tables.stops} SET ? WHERE id = ?`, [{
        stop_end: deviceData.timestamp,
        downtime: downtime,
        status: "Completed",
        updated_at: apiResponse.current_timestamp,
      }, stop_data.id]
    );

    return updateStop;
    
  } catch (error) {
    console.log("error in closing stop: ", error);
  }
};

const updateEvent = async (deviceData, apiResponse) => {
  try {
    const productionAtStart = parseFloat(apiResponse.current_event.production_at_start) > parseFloat(deviceData.total_production) ? 0 : parseFloat(apiResponse.current_event.production_at_start);

    const updateResult = await performQuery(
      `UPDATE ${tables.events} SET ? WHERE id= ?`,[{
        end_time: deviceData.timestamp,
        duration: (moment.utc(deviceData.timestamp) - moment.utc(apiResponse.current_event.start_time)) / 1000,
        speed: deviceData.machine_speed,
        total_production: parseFloat(deviceData.total_production) - parseFloat(productionAtStart),
        production_at_start: productionAtStart,
        production_at_end: deviceData.total_production,
        updated_at: apiResponse.current_timestamp,
      }, apiResponse.current_event.id]
    );
    return updateResult;
    
  } catch (error) {
    console.log("error in updating Event: ", error);
  }
};

const insertEvent = async (deviceData, apiResponse) => {
  try {
    const InsertResult =  await performQuery(
      `INSERT INTO ${tables.events} SET ?`, {
        line_id: deviceData.line_id,
        order_id: deviceData.order_id,
        shift_id: apiResponse.current_shift.id,
        user_id: apiResponse.current_user_id,
        stop_id: await getCurrentStopId(apiResponse),
        event_name: apiResponse.device_condition,
        start_time: deviceData.timestamp,
        end_time: deviceData.timestamp,
        duration: 0,
        speed: deviceData.machine_speed,
        total_production: 0,
        production_at_start: deviceData.total_production,
        production_at_end: deviceData.total_production,
        status: "In Progress",
        created_at: apiResponse.current_timestamp,
        updated_at: apiResponse.current_timestamp,
      }
    );

    return InsertResult;

  } catch (error) {
    console.log("error in inserting Event: ", error);
  }
};

const closePreviousEvents = async (deviceData, apiResponse) => {
  try {
    return await performQuery(
      `UPDATE ${tables.events} SET ? WHERE (line_id = ? AND id <> ? AND status = ?)`,
      [{
        status: 'Completed',
        updated_at: apiResponse.current_timestamp
      }, 
        deviceData.line_id, 
        apiResponse.new_event.id,
        'In Progress'
      ]
    );
    
  } catch (error) {
    console.log("error in closing previous order details: ", error);
  }
};


module.exports = async (workerData) => {
  try {
    // store device data from workerData
    const deviceData = workerData;
    
    // get System Time
    const systemTime = await getSystemTime();

    // Format current timestamp
    const currentTimestamp = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Store current Date
    const currentData = moment(systemTime).format("YYYY-MM-DD");

    // Timestamp from device data
    const timestamp = await getTimestamp(deviceData.timestamp);

    // get Current Active shift data
    const shiftData = await getShiftData(deviceData.timestamp);
    
    // get shift times 
    const shiftTimes = await getShiftTimes(shiftData?.[0], deviceData.timestamp);
    
    // get user id active in provided timestamp
    const attendanceData = await fetchAttendanceData(deviceData);

    
    // insert data in device_data and vfd_data table
    const insertedDeviceData = await insertDeviceData(deviceData, currentTimestamp);

    // store current timestamp in api response
    apiResponse.current_timestamp = currentTimestamp;

    // store current date in api response
    apiResponse.current_date = currentData;

    // store device data timestamp in api response
    apiResponse.device_data_timestamp = timestamp;

    // store device condition in api response
    apiResponse.device_condition = deviceData.status == 1 ? "Running" : "Stop";

    // store current user id in api response
    apiResponse.current_user_id = (attendanceData.length > 0 ? attendanceData[0]?.user_id : 0);

    // store inserted device data id in api response
    apiResponse.device_data_id= insertedDeviceData.insertId;

    // store current shift times in api response 
    apiResponse.current_shift = shiftTimes;

    // fetch existing active order details data 
    const orderDetailData = await fetchOrderDetails(deviceData, apiResponse);
    apiResponse.order_details = orderDetailData?.[0];
    apiResponse.current_order_details = apiResponse.order_details;

    // fetch existing active stop data
    const existingStopData = await fetchExistingStop(deviceData, apiResponse);
    apiResponse.existing_stop = existingStopData?.[0];
    apiResponse.current_stop = apiResponse.existing_stop;

    // fetch existing active event data
    const existingEventData = await fetchExistingEvent(deviceData, apiResponse);
    apiResponse.existing_event = existingEventData?.[0];
    apiResponse.current_event = apiResponse.existing_event;

    // Fetch Active Orders
    const activeOrders = await fetchActiveOrder(deviceData);
    apiResponse.active_order = activeOrders?.[0];
    
    // Check if order details exist
    if(orderDetailData?.length > 0) {
      // Update Order Details
      await updateOrderDetails(deviceData, apiResponse);
      apiResponse.order_details.production = deviceData.total_production;
      apiResponse.order_details.machine_speed = deviceData.machine_speed;
      apiResponse.order_details.order_end = deviceData.timestamp;
      apiResponse.current_order_details = apiResponse.order_details;
      apiResponse.message = "Order Details Updated";
    }
    else{
      // Insert Order Details
      apiResponse.new_order_details = {};
      const newCompleteOrdersData = await insertOrderDetails(deviceData, apiResponse);
      apiResponse.new_order_details.id = newCompleteOrdersData.insertId;
      apiResponse.new_order_details.production = deviceData.total_production;
      apiResponse.new_order_details.machine_speed = deviceData.machine_speed;
      apiResponse.new_order_details.order_start = deviceData.timestamp;
      apiResponse.new_order_details.order_end = deviceData.timestamp;
      apiResponse.current_order_details = apiResponse.new_order_details;
      apiResponse.message = "Order Details Created";

      // Close previous Order Details for the same line
      await closePreviousOrderDetails(deviceData, apiResponse);
    }

    // Manage Stops based on device condition
    if (apiResponse.device_condition === "Stop") {
      // Check if Stop Not exists
      if (existingStopData?.length !== 0) {
        // Update Existing Stop
        await updateStop(deviceData, apiResponse);
        apiResponse.existing_stop.stop_end = deviceData.timestamp;
        apiResponse.message = "Stop Updated";
      } else {
        // Create a new Stop
        apiResponse.newStopData = {};
        const newStopData = await createNewStop(deviceData, apiResponse);
        apiResponse.newStopData.id = newStopData.insertId;
        apiResponse.newStopData.stop_start = deviceData.timestamp;
        apiResponse.newStopData.stop_end = deviceData.timestamp;
        apiResponse.newStopData.downtime = 0;
        apiResponse.current_stop = apiResponse.newStopData;
        apiResponse.message = "New Stop Created";

        await closeStop(deviceData, apiResponse.newStopData);
        apiResponse.message = "Stop Closed";        
      }
    }
    else if (existingStopData?.length !== 0) {
      // Close Existing Stop
      await closeSpecificStop(deviceData, apiResponse.existing_stop);
      apiResponse.message = "Stop Closed";
    }

    // Manage Events
    // if Event Exists
    if (existingEventData?.length > 0) {
      // Update Previous Event
      const updatedEventData = await updateEvent(deviceData, apiResponse);
      apiResponse.current_event.end_time = deviceData.timestamp;
      apiResponse.current_event.duration = (moment.utc(deviceData.timestamp) - moment.utc(apiResponse.current_event.start_time)) / 1000;
      apiResponse.current_event.speed = deviceData.machine_speed;
      apiResponse.current_event.total_production = parseFloat(deviceData.total_production) - parseFloat(apiResponse.current_event.production_at_start);
      apiResponse.current_event.production_at_end = deviceData.total_production;
      apiResponse.message = "Event Updated";
    }
    else {
      // Create New Event 
      const newEventData = await insertEvent(deviceData, apiResponse);
      apiResponse.new_event = {};
      apiResponse.new_event.id = newEventData.insertId;
      apiResponse.new_event.event_name = apiResponse.device_condition;
      apiResponse.new_event.start_time = deviceData.timestamp;
      apiResponse.new_event.end_time = deviceData.timestamp;
      apiResponse.new_event.duration = 0;
      apiResponse.current_event = apiResponse.new_event;
      apiResponse.message = "New Event Created";

      // Close Previous Event if exists
      await closePreviousEvents(deviceData, apiResponse);
      apiResponse.message = "Previous Events Closed";
    }

    return {
      order_id: apiResponse.active_order?.id || 0,
      order_no: apiResponse.active_order?.order_no || '',
      shift_id: apiResponse.current_shift?.id || 0,
    };

  } catch (error) {
    console.log("error in worker.js: ", error);
    parentPort.postMessage({ error: error.message });
  }
};
